import sys
import os
import requests
import zipfile
import shutil
from PyQt5.QtGui import QIcon
from PyQt5.QtWidgets import QApplication, QMainWindow, QPushButton, QVBoxLayout, QWidget, QDialog, QLabel, QProgressBar, QHBoxLayout
from PyQt5.QtWebEngineWidgets import QWebEngineView
from PyQt5.QtCore import Qt, QThread, pyqtSignal, QUrl

SERVER_URL = "https://tinyapps.bolucraft.uk"
LOCAL_APPDATA = os.getenv("LOCALAPPDATA") or os.path.expanduser("~\\AppData\\Local")
BASE_DIR = os.path.join(LOCAL_APPDATA, "TinyAppsClient")
LOCAL_DIR = os.path.join(BASE_DIR, "local_files")
TEMP_DIR = os.path.join(BASE_DIR, "temp_files")
BACKUP_DIR = os.path.join(BASE_DIR, "backup")

for folder in [BASE_DIR, LOCAL_DIR, TEMP_DIR, BACKUP_DIR]:
    os.makedirs(folder, exist_ok=True)

os.environ["QTWEBENGINE_CHROMIUM_FLAGS"] = "--disable-gpu"
os.environ["QT_OPENGL"] = "software"

class DownloadThread(QThread):
    progress_changed = pyqtSignal(int)
    finished = pyqtSignal(bool, str)

    def run(self):
        try:
            zip_url = f"{SERVER_URL}/get"
            r = requests.get(zip_url, stream=True)
            total_length = int(r.headers.get('content-length', 0))
            zip_path = os.path.join(TEMP_DIR, "latest.zip")

            with open(zip_path, 'wb') as f:
                dl = 0
                for chunk in r.iter_content(chunk_size=8192):
                    if chunk:
                        f.write(chunk)
                        dl += len(chunk)
                        if total_length:
                            self.progress_changed.emit(int(dl / total_length * 100))

            temp_unpack_dir = os.path.join(TEMP_DIR, "unpack")
            if os.path.exists(temp_unpack_dir):
                shutil.rmtree(temp_unpack_dir)
            os.makedirs(temp_unpack_dir)
            with zipfile.ZipFile(zip_path, 'r') as zip_ref:
                zip_ref.extractall(temp_unpack_dir)

            if os.path.exists(BACKUP_DIR):
                shutil.rmtree(BACKUP_DIR)
            if os.path.exists(LOCAL_DIR):
                shutil.move(LOCAL_DIR, BACKUP_DIR)

            shutil.move(temp_unpack_dir, LOCAL_DIR)
            self.finished.emit(True, "Update erfolgreich!")
        except Exception as e:
            self.finished.emit(False, f"Update fehlgeschlagen: {e}")

class UpdateDialog(QDialog):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setWindowTitle("Update verfügbar")
        self.setFixedSize(360, 160)
        self.setStyleSheet("background-color: #2b2b2b; color: white;")

        self.layout = QVBoxLayout()
        self.info_label = QLabel("Versionisnfo: ...")
        self.info_label.setAlignment(Qt.AlignCenter)
        self.info_label.setStyleSheet("font-size: 14px;")
        self.progressbar = QProgressBar()
        self.progressbar.setValue(0)
        self.progressbar.setVisible(False)
        self.progressbar.setStyleSheet("""
            QProgressBar {
                border: 1px solid #555;
                border-radius: 8px;
                text-align: center;
                color: white;
            }
            QProgressBar::chunk {
                background-color: #2196F3;
                border-radius: 8px;
            }
        """)

        self.button_layout = QHBoxLayout()
        self.go_button = QPushButton("Go")
        self.cancel_button = QPushButton("Cancel")
        button_style = """
            QPushButton {
                background-color: #4CAF50;
                border-radius: 15px;
                color: white;
                font-size: 14px;
                padding: 6px 20px;
            }
            QPushButton:hover {
                background-color: #388E3C;
            }
        """
        self.go_button.setStyleSheet(button_style)
        self.cancel_button.setStyleSheet(button_style.replace("#4CAF50", "#f44336").replace("#388E3C", "#d32f2f"))

        self.button_layout.addWidget(self.go_button)
        self.button_layout.addWidget(self.cancel_button)

        self.layout.addWidget(self.info_label)
        self.layout.addWidget(self.progressbar)
        self.layout.addLayout(self.button_layout)
        self.setLayout(self.layout)

class MainWindow(QMainWindow):
    def __init__(self):
        super().__init__()
        self.setWindowTitle("TinyApps Client")
        self.resize(900, 600)
        self.webview = QWebEngineView()
        self.webview.setContextMenuPolicy(Qt.NoContextMenu)
        self.setCentralWidget(self.webview)

        self.update_btn = QPushButton("⟳", self)
        self.update_btn.setFixedSize(60, 60)
        self.update_btn.setStyleSheet("""
            QPushButton {
                background-color: #2196F3;
                border-radius: 30px;
                color: white;
                font-size: 24px;
            }
            QPushButton:hover {
                background-color: #1976D2;
            }
        """)
        self.update_btn.clicked.connect(self.show_update_popup)
        self.update_btn.show()

        self.home_btn = QPushButton("🏠", self)
        self.home_btn.setFixedSize(60, 60)
        self.home_btn.setStyleSheet("""
            QPushButton {
                background-color: #555555;
                border-radius: 30px;
                color: white;
                font-size: 24px;
            }
            QPushButton:hover {
                background-color: #777777;
            }
        """)
        self.home_btn.clicked.connect(self.load_local_index)
        self.home_btn.show()

        self.download_thread = DownloadThread()
        self.download_thread.progress_changed.connect(self.update_progress)
        self.download_thread.finished.connect(self.download_finished)

        self.update_dialog = UpdateDialog(self)
        self.update_dialog.go_button.clicked.connect(self.start_update)
        self.update_dialog.cancel_button.clicked.connect(self.update_dialog.reject)

        self.load_local_index()

    def resizeEvent(self, event):
        margin = 20
        self.update_btn.move(self.width() - 80, self.height() - 80)
        self.home_btn.move(self.width() - 150, self.height() - 80)
        super().resizeEvent(event)

    def load_local_index(self):
        index_path = os.path.join(LOCAL_DIR, "index.html")
        if os.path.exists(index_path):
            self.webview.load(QUrl.fromLocalFile(index_path))
        else:
            self.update_dialog.info_label.setText("Erste Version wird heruntergeladen...")
            self.download_thread.run()
            self.webview.load(QUrl.fromLocalFile(index_path))

    def show_update_popup(self):
        try:
            server_version = requests.get(f"{SERVER_URL}/version").text.strip()
        except:
            server_version = "unbekannte"
        local_version_file = os.path.join(LOCAL_DIR, "version.txt")
        local_version = "Keine"
        if os.path.exists(local_version_file):
            with open(local_version_file, "r") as f:
                local_version = f.read().strip()
        self.update_dialog.info_label.setText(f"Version {local_version} → {server_version}")
        self.update_dialog.progressbar.setVisible(False)
        self.update_dialog.progressbar.setValue(0)
        self.update_dialog.exec_()

    def start_update(self):
        self.update_dialog.progressbar.setVisible(True)
        self.download_thread.start()

    def update_progress(self, value):
        self.update_dialog.progressbar.setValue(value)

    def download_finished(self, success, message):
        self.update_dialog.progressbar.setVisible(False)
        self.update_dialog.info_label.setText(message)
        if success:
            self.webview.reload()
        else:
            if os.path.exists(BACKUP_DIR):
                if os.path.exists(LOCAL_DIR):
                    shutil.rmtree(LOCAL_DIR)
                shutil.move(BACKUP_DIR, LOCAL_DIR)
            self.webview.reload()

if __name__ == "__main__":
    app = QApplication(sys.argv)
    from pathlib import Path

    def resource_path(relative_path):
        if getattr(sys, 'frozen', False):
            base_path = os.path.dirname(sys.executable)
        else:
            base_path = os.path.dirname(__file__)
        return os.path.join(base_path, relative_path)
    icon_path = resource_path("tinyapps.ico")
    if os.path.exists(icon_path):
        app.setWindowIcon(QIcon(icon_path))
    window = MainWindow()
    window.show()
    sys.exit(app.exec_())
